------------------------------------------------------------------------------
-- Remote Monitor.lua -- Version 1.0
--
-- Utility script.
--
-- This script acts as a remote render monitor with Fusion 5.01
-- 
-- written by : sean konrad (sean@eyeonline.com)
-- written    : Nov. 15, 2005
-- 
------------------------------------------------------------------------------


-- Notes: iup list box entries are not numerically indexed like lua tables -- instead, they are assigned
-- string based properties that are.. numbers.  So, for example, the first entry in the iup list box named iupList 
-- would be printed by typing the following: print(iupList["1"]).

-- Because the lists are generated dynamically and display the slave / job info, what I've done is keep track of the 
-- number of slaves/ jobs and then use for loops with the iterating value converted to a string.

-- This gives definition to the numbers returned by the 'status attributes
-- for render jobs.  This has been converted to a string variable from a numeric index.

-- To make this work properly, we needed a new function to check to make sure that the manager still existed as 
-- repeatedly connecting was doing, well, bad things.  As such, this script won't function in 5.0.  You probably
-- shouldn't use the release build anyway.

-- Beyond that, the vast majority of the iup interface was created by playing around until I got something that 
-- I felt was functional.  Debugging iup is extremely difficult, as it doesn't always point you to what line the 
-- script is failing on and may on occasion do things with Fusion that aren't exactly kosher.  Save your work if
-- you're playing around with the design and functionality of an iup script.
ldofile(fusion:MapPath("Scripts:\\bmd.scriptlib"))

-- Create all of the buttons.  They will be positioned in the iup interface
-- later on in the code.
btn_refresh = iup.button{ title = "Refresh", size=80, FLAT="YES"}
btn_delete = iup.button{ title = "Delete Job", size = 80, FLAT="YES"}
btn_GroupsJob = iup.button{ title = "Modify Groups...", size = 80, FLAT="YES"}
textGroupsJob = iup.text{value = "", SIZE = "80",FGCOLOR = "200 200 200", MARGIN="15"}

-- Connect to the local fusion interface.
if not fusion then
	-- The second argument is the timeout.  Otherwise it will try to connect forever.
	fusion = Fusion("localhost", 10)
end
-- Set the master variable which will be used to connect to the render manager in question.
-- If there's no instance of Fusion, set it to connect to localhost (which it won't be able to do).  Then let the user specify.
if not fusion then 
	master = "localhost" 
else 
	master = fusion:GetPrefs()["Global"]["Network"]["ServerName"] 
	fusion:SetTimeout(0) 
end

-- Set up the connection button and its associated text box.  
btn_Connect = iup.button{ title = "Connect...", size = 80, FLAT="YES", MARGIN = "15"}
textConnect = iup.text{value = master, SIZE = "200",FGCOLOR = "200 200 200", MARGIN="15"}

btn_Frames = iup.button{ title = "Edit Frames", size = 80, FLAT="YES", MARGIN = "15"}
textFrames = iup.text{value ="", SIZE = "80",FGCOLOR = "200 200 200", MARGIN="15"}
btn_User = iup.button{ title = "Change User", size = 80, FLAT="YES", MARGIN = "15"}
textUser = iup.text{value ="", SIZE = "80",FGCOLOR = "200 200 200", MARGIN="15"}

function getfilename(path)
	for i = string.len(path), 1, -1 do
		teststring = string.sub(path, i, i)
		if teststring == "\\" or teststring == "/" then
			return string.sub(path, i+1)
		end
	end
end



-- This will format the job's attributes so that it can be displayed in the iup list box.
-- This is having the same problem as the above function since its conversion to the iup format.

function fmtJob(job)
	a = job:GetAttrs()

	-- Get the total number of frames by adding the Unrendered, Rendered, and Rendering frames.
    total = a.RJOBN_UnrenderedFrames + a.RJOBN_RenderedFrames + a.RJOBN_RenderingFrames
	
	--Also get the total number of rendered frames.
    done  = a.RJOBN_RenderedFrames

	if a.RJOBS_PriorityClass == nil then
	   classes = nil
	else
	   classes = a.RJOBS_PriorityClass
	end
  
	if a.RJOBS_Type == "Comp" then
		-- Here we use string.format to define the spacing between job names.
		-- Incidentally, the only reason this works is because we're using a fixed width font
		-- in the list box (Lucida Console).  
		local strAssembly = string.format("%-35.35s", bmd.parseFilename(a.RJOBS_Name).FullName).." "..
		string.format("%-16.16s", a.RJOBS_QueuedBy).." "..
		string.format("%-15.15s", done.."/"..total).." "..
		string.format("%-15.15s", a.RJOBS_Status).." "
		if classes then
			strAssembly=strAssembly..string.format("%-10.10s", classes)
		end
		return strAssembly
	end
end

-- This function was added in to check if the host server is still actually connected.
-- If it isn't, error out.

function connectCheck()
	if masterConnect then 
		if masterConnect:IsAppConnected() == true then 
			
			return true
			
		end
	end
	errorConnect()
	return nil
end

-- Error if the host is no longer connected.

function errorConnect()
	
	clear_j()
	j["1"] = "Could not connect to ".. master
end


-- This will first clear and then refill both of the list boxes.
function fillList()
	clear_j()
	fill_j()
end

-- Clears the list box named "J"  
function clear_j()
	if jl then
		for i, v in pairs(jl) do
			j[tostring(i)]=nil
		end
	end
	-- If there's an error message in the list NOT part of the list box, clear it.
	if j["1"] then
		j["1"] = nil
	end
end

function fill_j()
	jl = {}
	-- Fill the job list box.
	-- Check the connection.
	if connectCheck() then
        local cnt = 0
		temp_jl = rm:GetJobList()
		for i, v in pairs(temp_jl) do
            if v:GetAttrs().RJOBS_QueuedBy == user then
    			cnt = cnt + 1
    			j[tostring(i)]=fmtJob(v)
    			jl[cnt] = v
            end
		end
	end
end



-- This is the primary function used to connect to the master.  It's also used to create the list boxes if they haven't been created yet.

function connectMaster()

	-- a boolean is defined to keep track of whether or not the list boxes have been created yet.
	if listsCreated then
	   
		-- if they've already been created, they obviously want to change which master they're connecting to.
		master = textConnect.value
		
		-- connect to an instance of fusion -- defined by the 'master' variable.  
		-- set the timeout to 5, and make sure we connect to a Render manager first.  
		-- if that fails, connect to the main instance of Fusion.
		
		-- we specify RenderManager, because there's a chance that it might connect to another 
		-- instance of Fusion first.  depending on which was opened first.
		masterConnect = Fusion(master, 5.0, nil, "RenderManager") 
		if masterConnect == nil then
			masterConnect = Fusion(master, 5.0) 
		end
		
		-- if it's connected display the lists.
		if connectCheck() == true then
			masterConnect:SetTimeout(0)
			-- also, define the rm variable.
			rm = masterConnect.RenderManager
			refreshMonitor()
		end
	else
        if not user then user = fusion:GetEnv("USERNAME") end
		textUser.value = user
		-- set the listsCreated boolean to true.
		listsCreated = true
		
		-- Define the size of the lists and create them using the iup.list() function...
		j = iup.list{SIZE="405x180"}
		
		-- Deselect list box items.
		j.value = 0
		
		-- Set the BG color to something lighter than the main interface to give it contrast.
		j.BGCOLOR = "75 75 75"
		
		-- the font is lucide console because it a fixed width font.
		j.FONT = "lucida console::"
		

		
		-- Connect
		masterConnect = Fusion(master, 5.0, nil, "RenderManager") 
		if masterConnect == nil then
			masterConnect = Fusion(master, 5.0) 
		end
		if masterConnect then
			masterConnect:SetTimeout(0)
			rm = masterConnect.RenderManager
			fillList()
		else
			errorConnect()
		end
	end
end


-- Connect!! why not?
connectMaster()


----------------------------------------------------------------------------------------------------------------------------------------------------
-- This section is for spawning the interface of both the main window and the popup windows..-------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

-- Using a vbox will make all items in it be aligned vertically.  If you're interested in further exposition about the interface
-- features of iup, it's suggested you read the actual documentation available on the iup website.  
vbox1 =   iup.vbox
  {
    iup.hbox
    {
               
        iup.vbox
        {
			-- buttons and groups text box go here.  FGCOLOR of the text is the off white colour.
			btn_refresh,
			btn_delete,
			
			-- We define a label here, because its settings are minimal.
			iup.label{title = "Job's Groups:", FGCOLOR = "200 200 200"},
			textGroupsJob,
			btn_GroupsJob,
			iup.label{title = "Job's Frames:", FGCOLOR = "200 200 200"},
			textFrames,
			btn_Frames,
			iup.label{title = "Current User:", FGCOLOR = "200 200 200"},
			
			textUser,
			btn_User,
			SIZE = "180"
        },
        iup.vbox
        {
			iup.label{title = "Job List:", FGCOLOR = "200 200 200"},
			j
		},     
	-- Sets a 5 point gap between interface items.
	GAP = "5"
	},
	iup.vbox{iup.hbox{}, GAP = "0",
	--iup.hbox
		--{ btn_Frames,
		--textFrames
		--;GAP = "15", SIZE = "180x90", ALIGNMENT = "ACENTER" 
		--},
	iup.hbox
		{ btn_Connect,
		textConnect, GAP = "15", SIZE = "180x90", ALIGNMENT = "ACENTER" 
		}}, MARGIN = "5x5"}
    
  
dlg = iup.dialog
{
  vbox1, title="IUP Egocentric Render Monitor", menu=mnu, SIZE="680x215", BGCOLOR="60 60 60", FGCOLOR = "200 200 200", TOPMOST = "NO", FONT = "", RESIZE = "NO", MAXBOX = "NO"
}


----------------------------------------------------------------------------------------------------------------------------------------------------
-- This section is for functions related to file browsing for a .comp file to add to the manager.---------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

function refreshMonitor()
	
	-- The usual step for anything related to the manipulation of the queue is to check if it's conected...
	
	if connectCheck() then
		
		-- if it is, clear the boxes and refill them
		clear_j()
		fill_j()
		
		-- clear the text boxes as nothing will be highlighted anymore.
		textGroupsJob.value = ""
		textFrames.value =""
	end
end

----------------------------------------------------------------------------------------------------------------------------------------------------
------------- This area of the code exists primarily as an area for button press related functions -------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------


function btn_refresh:action()
	-- Refill the list boxes.  The function refreshMonitor will check to make sre that it's still connected to Fusion
	-- for us, so no need to call it now.
	refreshMonitor()
	
  	return iup.DEFAULT
end


-- This is the action for the button that deletes the selected job from the queue.
function btn_delete:action()
	-- if the list even has an entry in it..
	if j["1"] then
	
		-- if something is selected...
		if j.value ~=0 then
		
			-- if it's connected ..
			if connectCheck() then 
				
				-- Get the job list to see if it still matches the old job list (in case the job list somehow got altered -- job already deleted, etc.).
				jl_new = rm:GetJobList()
				
				-- Get the currently selected entry -- convert it to a number value so that it can actually be used 
				-- when we're iterating through tables..
				cur_Sel = tonumber(j.value)
				
				-- Compare the old list to the new..
				if tostring(jl[cur_Sel]) == tostring(jl_new[cur_Sel]) then
					
					-- Display a message in the monitor with using the Log() function..
					rm:Log("REMOTE IUP MONITOR job removed "..jl[cur_Sel]:GetAttrs().RJOBS_Name)
					
					-- Remove the job
					rm:RemoveJob(jl_new[cur_Sel])
					
					-- Give it a second to delete.
					wait(1)
					
					-- Refresh.
					refreshMonitor()
				else
				
					-- If the entries no longer match, display an error.
					clear_j()
					j["1"] = "Queue manipulated since last update -- please refresh and try again."
				end
			end
		end
	end
	return iup.DEFAULT
end


-- Holla back (see above).
function btn_GroupsJob:action()
	if j["1"] then
		if j.value ~=0 then
			if connectCheck() then
				-- jl_new = rm:GetJobList()
				cur_Sel = tonumber(j.value)
				--if tostring(jl[cur_Sel]) == tostring(jl_new[cur_Sel]) then
					rm:Log("REMOTE IUP MONITOR modified Groups of job : "..jl_new[cur_Sel]:GetAttrs().RJOBS_Name)
					jl[cur_Sel]:SetAttrs({RJOBS_Groups = textGroupsJob.value})
					
					wait(1)
					refreshMonitor()
				--else
					clear_j()
					j["1"] = "Queue manipulated since last update -- please refresh and try again."
				--end
			end
		end
	end
	return iup.DEFAULT
end

-- added in bit to change frame ranges
function btn_Frames:action()
	if j["1"] then
		if j.value ~=0 then
			if connectCheck() then
				-- jl_new = rm:GetJobList()
				cur_Sel = tonumber(j.value)
				--if tostring(jl[cur_Sel]) == tostring(jl_new[cur_Sel]) then
					rm:Log("REMOTE IUP MONITOR modified Frames of job : "..jl[cur_Sel]:GetAttrs().RJOBS_QueuedBy)
					jl[cur_Sel]:SetFrames(textFrames.value)
					
					wait(1)
					refreshMonitor()
				--else
					--clear_j()
					--j["1"] = "Queue manipulated since last update -- please refresh and try again."
				--end
			end
		end
	end
	return iup.DEFAULT
end

-- sets the user name...
function btn_User:action()
	user = textUser.value
	wait(.5)
	refreshMonitor()
end
----------------------------------------------------------------------------------------------------------------------------------------------------
------------- This area of the code exists primarily as an area for list click related functions ---------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

-- When one clicks in the job list, it will change what's displayed in the groups text box. 
function j:action()
	if connectCheck() then		
		if j["1"] then
		
			-- if it's connected, then change the text box to equal the RJOBS_Groups attribute
			textGroupsJob.value = jl[tonumber(j.value)]:GetAttrs().RJOBS_Groups
			textFrames.value = jl[tonumber(j.value)]:GetFrames()
		end
	end
end


----------------------------------------------------------------------------------------------------------------------------------------------------
--------------- Connect button. --------------------------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

function btn_Connect:action()
	connectMaster()
end



----------------------------------------------------------------------------------------------------------------------------------------------------
--------------- Automatic Refresh. -----------------------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

-- One of the nice features of IUP is the ability to create a timer
-- that will call an action every X units of time (milliseconds).
timer = iup.timer{time=300000.0, run="YES"}

 function timer:action_cb()
	if connectCheck() then
		refreshMonitor()
		return iup.DEFAULT
	else
		connectMaster()
	end
 end 


-- Show the interface
dlg:showxy(iup.CENTER, iup.CENTER)
-- Throw it into the main loop.
iup.MainLoop()
